using System;
using System.Collections;

using Microsoft.DirectX;
using Microsoft.DirectX.Direct3D;

using DarkStrideToolbox;


namespace DarkStride.StellarLanes.SharedDLL
{
	public class StellarLanesParticleManager : DSParticleManager
	{
        #region Properties
        private bool m_bVaryParticleLifeSpan = true;
        private Session m_oSession = null;
        private string m_sZoneID = string.Empty;
        #endregion

        public override void CreateNewParticle(DSParticle oParticle)
        {
            base.CreateNewParticle(oParticle);
            if (this.VaryParticleLifeSpan == true)
            {
                oParticle.m_dLifeTimeOfParticle *= DSMisc.GetRnd(.8, 1.2);
            }
        }
        public override void Render(Device o3DDevice, Vector2 oViewUpperLeft)
        {
            const string sRoutineName = "DarkStrideToolbox.ParticleManager.Render";
            //string sCaptureBlockKey = "Particle Capture Block Key";
            PointVertex[] vertices = null;
            int numParticlesToRender = 0;
            int count = 0;
            Vector2 vNewPPos = Vector2.Empty;
            Vector2 vVel = Vector2.Empty;
            Vector3 vPos = Vector3.Empty;
            float fLengthSq;
            Texture oTexture = null;
            Session oSession = null;
            RenderSettings oRenderSettings = null;


            try
            {
                if (m_sZoneID != String.Empty && Globals.Inst().MyZone.ZoneID.ToString() != m_sZoneID)
                {
                    return;
                }

                o3DDevice.SetTransform(TransformType.World, Matrix.Identity);
                o3DDevice.SetTransform(TransformType.View, Matrix.Identity);
                o3DDevice.SetTransform(TransformType.Projection, Matrix.OrthoOffCenterLH(
                                0, Globals.Inst().GameEngine.ScreenWidth,
                                Globals.Inst().GameEngine.ScreenHeight, 0, 0, 1000));


                oSession = Globals.Inst().Session;
                oRenderSettings = oSession.LastRenderSettingsUsed;
                if (oSession.AreaToDrawAt == System.Drawing.Rectangle.Empty)
                {
                    return;
                }

                if (base.ParticleTexture != null && base.ParticleTexture.Length > 0)
                {
                    oTexture = DSResourceManager.GetGlobalInstance().GetTexture(base.ParticleTexture);
                }
                else
                {
                    oTexture = DSResourceManager.GetGlobalInstance().GetTexture("System_Particle");
                }
                o3DDevice.SetTexture(0, oTexture);

                // Set up the vertex buffer to be rendered
                o3DDevice.SetStreamSource(0, base.VertexBuffer, 0);
                o3DDevice.VertexFormat = PointVertex.Format;

                o3DDevice.RenderState.ZBufferWriteEnable = false;
                o3DDevice.RenderState.AlphaBlendEnable = true;
                o3DDevice.RenderState.SourceBlend = Blend.One;
                o3DDevice.RenderState.DestinationBlend = Blend.One;

                //Set the render states for using point sprites
                o3DDevice.RenderState.PointSpriteEnable = true;
                o3DDevice.RenderState.PointScaleEnable = false;
                o3DDevice.RenderState.PointSize = base.ParticleSize * (float)Globals.Inst().Session.CurrentZoom;
                o3DDevice.RenderState.PointSizeMin = 0.00f;
                o3DDevice.RenderState.PointScaleA = 0.00f;
                o3DDevice.RenderState.PointScaleB = 0.00f;
                o3DDevice.RenderState.PointScaleC = 1.00f;

                o3DDevice.SetTextureStageState(0, TextureStageStates.ColorOperation, (int)TextureOperation.Modulate); // Modulate...
                o3DDevice.SetTextureStageState(0, TextureStageStates.ColorArgument1, (int)TextureArgument.TextureColor);   // the texture for this stage with...
                o3DDevice.SetTextureStageState(0, TextureStageStates.ColorArgument2, (int)TextureArgument.Diffuse);   // the diffuse color of the geometry.
                o3DDevice.SetTextureStageState(0, TextureStageStates.AlphaOperation, (int)TextureOperation.SelectArg1);
                o3DDevice.SetTextureStageState(0, TextureStageStates.AlphaArgument1, (int)TextureArgument.TextureColor);
                o3DDevice.SetTextureStageState(1, TextureStageStates.ColorOperation, (int)TextureOperation.Disable);
                o3DDevice.SetTextureStageState(1, TextureStageStates.AlphaOperation, (int)TextureOperation.Disable);


                // Lock the vertex buffer.  We fill the vertex buffer in small
                // chunks, using LockFlags.NoOverWrite.  When we are done filling
                // each chunk, we call DrawPrim, and lock the next chunk.  When
                // we run out of space in the vertex buffer, we start over at
                // the beginning, using LockFlags.Discard.
                base.BaseParticle += base.Flush;

                if (base.BaseParticle >= base.Discard)
                    base.BaseParticle = 0;

                vertices = (PointVertex[])base.VertexBuffer.Lock(base.BaseParticle * DXHelp.GetTypeSize(typeof(PointVertex)), typeof(PointVertex), (base.BaseParticle != 0) ? LockFlags.NoOverwrite : LockFlags.Discard, base.Flush);
                foreach (DSParticle oLoopParticle in base.ParticlesList)
                {
                    //11/19/2004 Chris Hill  This code was assuming the viewport was 500x500!!
                    vNewPPos = new Vector2(oLoopParticle.m_vPosition.X, oLoopParticle.m_vPosition.Y);
                    vNewPPos = oSession.ConvertWorldPtToScreenPt(oRenderSettings, vNewPPos);

                    vPos = new Vector3(vNewPPos.X, vNewPPos.Y, 0);

                    vVel = oLoopParticle.m_vVelocity;
                    fLengthSq = vVel.LengthSq();

                    vertices[count].v = vPos;
                    vertices[count].color = GetParticleColor(oLoopParticle);
                    count++;

                    if (++numParticlesToRender == base.Flush)
                    {
                        // Done filling this chunk of the vertex buffer.  Lets unlock and
                        // draw this portion so we can begin filling the next chunk.
                        base.VertexBuffer.Unlock();
                        o3DDevice.DrawPrimitives(PrimitiveType.PointList, base.BaseParticle, numParticlesToRender);

                        // Lock the next chunk of the vertex buffer.  If we are at the 
                        // end of the vertex buffer, LockFlags.Discard the vertex buffer and start
                        // at the beginning.  Otherwise, specify LockFlags.NoOverWrite, so we can
                        // continue filling the VB while the previous chunk is drawing.
                        base.BaseParticle += base.Flush;

                        if (base.BaseParticle >= base.Discard)
                        {
                            base.BaseParticle = 0;
                        }

                        vertices = (PointVertex[])base.VertexBuffer.Lock(base.BaseParticle * DXHelp.GetTypeSize(typeof(PointVertex)), typeof(PointVertex), (base.BaseParticle != 0) ? LockFlags.NoOverwrite : LockFlags.Discard, base.Flush);
                        count = 0;

                        numParticlesToRender = 0;
                    }
                }


                // Unlock the vertex buffer
                base.VertexBuffer.Unlock();
                // Render any remaining m_nNumParticles
                if (numParticlesToRender > 0)
                    o3DDevice.DrawPrimitives(PrimitiveType.PointList, base.BaseParticle, numParticlesToRender);


                // Reset render states
                o3DDevice.RenderState.PointSpriteEnable = false;
                o3DDevice.RenderState.PointScaleEnable = false;
            }
            catch (System.Exception oEx)
            {
                throw new System.Exception(sRoutineName + " Failed.", oEx);
            }
        }

        #region Properties
        public bool VaryParticleLifeSpan
        {
            get
            {
                return (m_bVaryParticleLifeSpan);
            }
            set
            {
                m_bVaryParticleLifeSpan = value;
            }
        }
        public string ZoneID
        {
            get
            {
                return (m_sZoneID);
            }
            set
            {
                m_sZoneID = value;
            }
        }
        public Session Session
        {
            get
            {
                return (m_oSession);
            }
            set
            {
                m_oSession = value;
            }
        }
        #endregion
	}

    public class SLEntityEffect : StellarLanesParticleManager
	{
		#region Properties
		private bool m_bCircleThrustLine = false;

        private Entity m_oEntityToCenterOn = null;
		#endregion


		public SLEntityEffect(){}
        public override void Advance(double dElapsedTime)
        {
            if (m_oEntityToCenterOn != null)
            {
                this.EmitterPosition = m_oEntityToCenterOn.Pos;
                this.EmittersVelocity = m_oEntityToCenterOn.Vel;
            }

            base.Advance(dElapsedTime);
        }

        public static SLEntityEffect StartMissileThrust(Vector2 vPos, Vector2 vDir, string sZoneID)
		{
			SLEntityEffect oPMan = null;


			oPMan = new SLEntityEffect();

			oPMan.Colors = new System.Drawing.Color[]{	System.Drawing.Color.White,
														System.Drawing.Color.Red,
														System.Drawing.Color.LightGray,//Gray,
														System.Drawing.Color.LightGray,// .Gray,
														System.Drawing.Color.Black };
			oPMan.PercTimeInColor = new double[]{ .01,.02,.1,.7,1 };
            oPMan.EmitVelocity = 100;// 5;
			oPMan.NumParticlesToEmit = 200;
			oPMan.EmitterPosition = vPos;
			oPMan.EmitterDirection = vDir;
			oPMan.EmittersVelocity = vDir * -1;
			oPMan.EmitterRangeInRad = Math.PI / 4.0;
			oPMan.ParticleLifeTimeInSeconds = 2.0;
            oPMan.ParticleSize = 8;
			oPMan.CircleThrustLine = false;
            oPMan.ZoneID = sZoneID;

			Globals.Inst().GameEngine.ParticleSystem.CreateParticleSystem( oPMan );
			

			return( oPMan );
		}
        public static SLEntityEffect StartShipThrusterThrust(Vector2 vPos, Vector2 vDir, string sZoneID)
		{
			SLEntityEffect oPMan = null;


			oPMan = new SLEntityEffect();

			oPMan.Colors = new System.Drawing.Color[]{	System.Drawing.Color.White,
														System.Drawing.Color.Red,
														System.Drawing.Color.LightGray,
														System.Drawing.Color.LightGray,
														System.Drawing.Color.Black };
			oPMan.PercTimeInColor = new double[]{ .01,.02,.1,.7,1 };
			oPMan.EmitVelocity = 50;
			oPMan.NumParticlesToEmit = 100;
			oPMan.EmitterPosition = vPos;
			oPMan.EmitterDirection = vDir;
			oPMan.EmittersVelocity = vDir * -1;
			oPMan.EmitterRangeInRad = Math.PI / 3.0;
			oPMan.ParticleLifeTimeInSeconds = 5.0;
            oPMan.ParticleSize = 35;
			oPMan.CircleThrustLine = false;
            oPMan.ZoneID = sZoneID;

			Globals.Inst().GameEngine.ParticleSystem.CreateParticleSystem( oPMan );
			

			return( oPMan );
		}
        public static SLEntityEffect StartClassicShipThrusterThrust(Vector2 vPos, Vector2 vDir, string sZoneID)
		{
			SLEntityEffect oPMan = null;


			oPMan = new SLEntityEffect();

			oPMan.Colors = new System.Drawing.Color[]{	System.Drawing.Color.Red,
														System.Drawing.Color.Yellow,
														System.Drawing.Color.Black };
			oPMan.PercTimeInColor = new double[]{ .5,.5,1 };
			oPMan.EmitVelocity = .1f;
			oPMan.NumParticlesToEmit = 8;
			oPMan.EmitterPosition = vPos;
			oPMan.EmitterDirection = vDir;
			oPMan.EmittersVelocity = vDir * -1;
			oPMan.EmitterRangeInRad = Math.PI / 3.0;
			oPMan.ParticleSize = 25;
			oPMan.CircleThrustLine = false;
			oPMan.ParticleTexture = "Classic_Thruster_Trail_Particle";
			oPMan.ParticleLifeTimeInSeconds = 1.1f;
            oPMan.ZoneID = sZoneID;

			Globals.Inst().GameEngine.ParticleSystem.CreateParticleSystem( oPMan );
			

			return( oPMan );
		}
        public static SLEntityEffect StartAIThrusterThrust(Vector2 vPos, Vector2 vDir, string sZoneID)
		{
			SLEntityEffect oPMan = null;


			oPMan = new SLEntityEffect();

			oPMan.Colors = new System.Drawing.Color[]{	System.Drawing.Color.White,
														 System.Drawing.Color.Red,
														 System.Drawing.Color.LightGray,//Gray,
														 System.Drawing.Color.LightGray,//Gray,
														 System.Drawing.Color.Black };
			oPMan.PercTimeInColor = new double[]{ .01,.02,.1,.7,1 };
			oPMan.NumParticlesToEmit = 100;
			oPMan.EmitterPosition = vPos;
			oPMan.EmitterDirection = vDir;
			oPMan.EmittersVelocity = vDir * -1;
			oPMan.EmitterRangeInRad = Math.PI / 3.0;
			oPMan.ParticleLifeTimeInSeconds = 1.0;
			oPMan.ParticleSize = 15;
			oPMan.EmitVelocity = 8;
			oPMan.CircleThrustLine = false;
            oPMan.ZoneID = sZoneID;

			Globals.Inst().GameEngine.ParticleSystem.CreateParticleSystem( oPMan );
			

			return( oPMan );
		}

        public static SLEntityEffect StartLevelupEffect(Entity oEntity)
        {
            SLEntityEffect oPMan = null;


            oPMan = new SLEntityEffect();

            oPMan.Colors = new System.Drawing.Color[]{	System.Drawing.Color.White,
														 System.Drawing.Color.Yellow,
														 System.Drawing.Color.Orange,
														 System.Drawing.Color.Black };
            oPMan.PercTimeInColor = new double[] { .1, .2, .2, .5 };
            oPMan.NumParticlesToEmit = 400;
            oPMan.EmitterPosition = oEntity.Pos;
            oPMan.EmitterDirection = new Vector2(0, 0);
            oPMan.EmittersVelocity = oEntity.Vel;
            oPMan.EmitterRangeInRad = Math.PI * 2.0;
            oPMan.ParticleLifeTimeInSeconds = .5;
            oPMan.ParticleSize = 60;
            oPMan.EmitVelocity = 400;
            oPMan.CircleThrustLine = false;
            oPMan.ParticleManagerLifeTimeInSeconds = .5;
            oPMan.VaryParticleLifeSpan = false;
            oPMan.ZoneID = oEntity.Location.ZoneID.ToString();

            oPMan.EntityToCenterOn = oEntity;
            Globals.Inst().GameEngine.ParticleSystem.CreateParticleSystem(oPMan);


            return (oPMan);
        }


		#region Properties
		public bool CircleThrustLine
		{
			get
			{
				return( m_bCircleThrustLine );
			}
			set
			{
				m_bCircleThrustLine = value;
			}
		}

        public Entity EntityToCenterOn
        {
            get
            {
                return (m_oEntityToCenterOn);
            }
            set
            {
                m_oEntityToCenterOn = value;
            }
        }
		#endregion
	}
    /* This has a sun like effect on this particle system
     oPMan.Colors = new System.Drawing.Color[]{	System.Drawing.Color.Purple,
                                                            System.Drawing.Color.White,
                                                            System.Drawing.Color.Yellow,
													        System.Drawing.Color.Black }* 
    */
    public class SLPortalEffect : StellarLanesParticleManager
    {
        #region Properties
        private Entity m_oEntityToCenterOn = null;
        #endregion


        public SLPortalEffect(){}
        public override void Advance(double dElapsedTime)
        {
            if (m_oEntityToCenterOn != null)
            {
                this.EmitterPosition = m_oEntityToCenterOn.Pos;
                this.EmittersVelocity = m_oEntityToCenterOn.Vel;
            }

            base.Advance(dElapsedTime);
        }

        public static SLPortalEffect StartPortalEffect(Session oSession, Vector2 vPos, string sZoneID, bool bActiveColors)
        {
            SLPortalEffect oPMan = null;


            oPMan = new SLPortalEffect();
            oPMan.Session = oSession;

            if (bActiveColors == true)
            {
                oPMan.Colors = new System.Drawing.Color[]{	System.Drawing.Color.DarkBlue,
														    System.Drawing.Color.Yellow,
														    System.Drawing.Color.Purple,
														    System.Drawing.Color.Black };
            }
            else
            {
                oPMan.Colors = new System.Drawing.Color[]{	System.Drawing.Color.Red,
                                                            System.Drawing.Color.White,
                                                            System.Drawing.Color.Maroon,
													        System.Drawing.Color.Black };
            }
            oPMan.PercTimeInColor = new double[] { .1, .2, .2, .5 };
            oPMan.NumParticlesToEmit = 400;
            oPMan.EmitterPosition = vPos;
            oPMan.EmitterDirection = new Vector2(0, 0);
            oPMan.EmitterRangeInRad = Math.PI * 2.0;
            oPMan.ParticleLifeTimeInSeconds = 2.0;
            oPMan.ParticleSize = 80;
            oPMan.EmitVelocity = 200;
            oPMan.ZoneID = sZoneID;

            Globals.Inst().GameEngine.ParticleSystem.CreateParticleSystem(oPMan);


            return (oPMan);
        }
        public static SLPortalEffect StartPortalDestTransmitEffect(Session oSession, Vector2 vPos, Vector2 vVel, string sZoneID)
        {
            SLPortalEffect oPMan = null;


            oPMan = new SLPortalEffect();
            oPMan.Session = oSession;
            oPMan.Colors = new System.Drawing.Color[]{	System.Drawing.Color.Black,
                                                        System.Drawing.Color.White,
                                                        System.Drawing.Color.Purple,
													    System.Drawing.Color.Black };
            oPMan.PercTimeInColor = new double[] { .1, .2, .8, 1 };
            oPMan.NumParticlesToEmit = 200;
            oPMan.EmitterPosition = vPos;
            oPMan.EmitterDirection = vVel;
            oPMan.EmitterRangeInRad = Math.PI / 4.0;
            oPMan.ParticleLifeTimeInSeconds = 2;
            oPMan.ParticleSize = 50;
            oPMan.EmitVelocity = vVel.Length() / 3f;
            oPMan.ParticleManagerLifeTimeInSeconds = 1;
            oPMan.ZoneID = sZoneID;

            Globals.Inst().GameEngine.ParticleSystem.CreateParticleSystem(oPMan);


            return (oPMan);
        }
    }
    public class SLPortalEffectSourceTransmit : SLPortalEffect
    {
        #region Member Variables
        private const double m_cEFFECTTIME = .9;
        private const double m_cNUMPARTICLES_ATEND = 500;
        private const double m_cNUMPARTICLES_BASE = 50;

        private double m_nRadius = 0;
        private double m_nRadiusGrowth = 800;
        #endregion


        public static SLPortalEffectSourceTransmit StartPortalSourceTransmitEffect(Session oSession, Vector2 vPos, string sZoneID)
        {
            SLPortalEffectSourceTransmit oPMan = null;


            oPMan = new SLPortalEffectSourceTransmit();
            oPMan.Session = oSession;
            oPMan.Colors = new System.Drawing.Color[]{	System.Drawing.Color.Purple,
                                                        System.Drawing.Color.White,
                                                        System.Drawing.Color.Yellow,
													    System.Drawing.Color.Black };
            oPMan.PercTimeInColor = new double[] { .2, .4, .7, 1 };
            oPMan.NumParticlesToEmit = m_cNUMPARTICLES_BASE;
            oPMan.EmitterPosition = vPos;
            oPMan.EmitterDirection = new Vector2(0, 0);
            oPMan.EmitterRangeInRad = Math.PI * 2.0;
            oPMan.ParticleLifeTimeInSeconds = 2.0;
            oPMan.ParticleSize = 120;
            oPMan.ParticleManagerLifeTimeInSeconds = m_cEFFECTTIME;
            oPMan.ZoneID = sZoneID;

            Globals.Inst().GameEngine.ParticleSystem.CreateParticleSystem(oPMan);


            return (oPMan);
        }

        public override void Advance(double dElapsedTime)
        {
            base.Advance(dElapsedTime);

            m_nRadius += dElapsedTime * m_nRadiusGrowth;
            this.NumParticlesToEmit = m_cNUMPARTICLES_BASE + m_cNUMPARTICLES_ATEND * (1 - (this.CurrentTime / m_cEFFECTTIME));
        }
        public override void CreateNewParticle(DSParticle oParticle)
        {
            double nRad = 0;


            base.CreateNewParticle(oParticle);

            //Adjust our starting position
            nRad = DSMisc.GetRnd(0, Math.PI * 2);
            oParticle.m_vPosition = new Vector2(
                                this.EmitterPosition.X + (float)(Math.Cos(nRad) * m_nRadius),
                                this.EmitterPosition.Y + (float)(Math.Sin(nRad) * m_nRadius));
            oParticle.m_vVelocity = new Vector2(
                                (float)(Math.Cos(nRad) * m_nRadiusGrowth * .7),
                                (float)(Math.Sin(nRad) * m_nRadiusGrowth * .7));
        }
    }
}
